(* ::Subsubsection:: *)
(*SchmidtCassegrainCAfter package*)

(* ::Text:: *)
(*This impements the Schmidt-Cassegrain telescope with corrector next to the focus of a Cassegrain combination*)

(* ::Text:: *)
(*Chapter*)

(* ::Text:: *)
(*This file should be in the following path*)

(* ::Text:: *)
(*FileNameJoin[{$UserBaseDirectory, "Applications/GeometricOptics/SchmidtCassegrainCAfter"}]*)

BeginPackage["GeometricOptics`Packages`SchmidtCassegrainCAfter`", {"GeometricOptics`"}]

Options[SchmidtCassegrainCAfter] = {OutputType->"Report", OutputLevel->"Full"};
(* 	OutputType can be 
					"Report", generates a new report (notebook) for each call to the function
					"Print", prints results in the current position of the evaluating notebook
					"Basic", gives the list of results
					"Values" or any other directive, gives the list of values (no names of variables)
	OutputLevel can be
					"Minimal", gives only output values, those calculated by the function
					"Full", gives the list of input parameters and output values
	CameraType can be
					"SD" Houghton's telescope with symmetric doublet
					"DDR" Houghton's telescope with doublet with different radii - no control of astigmatism
*)

SchmidtCassegrainCAfter::usage="SchmidtCassegrainCAfter calculates parameters for a Houghton-Cassegrain telescope optical system.";
SchmidtCassegrainCAfterInputPanel::usage = "SchmidtCassegrainCAfterInputPanel gives the GUI panel to work with SchmidtCassegrainCAfter";

$SchmidtCassegrainCAfterInputVariables = {"f1", "ft", "em", "diam", "\[Theta]"};

Begin["`Private`"]

SchmidtCassegrainCAfter[{f1_, ft_, em_, diam_, theta_}, opts___]:= SchmidtCassegrainCAfter[f1, ft, em, diam, theta];

SchmidtCassegrainCAfter[f1_, Ni_, em_, diam_, theta_, OptionsPattern[]] := 
Quiet@Module[{\[Alpha], \[Beta], \[Gamma], rad, thick, ind, costasf, s3, K1, K2, a4, as, Rc, inputs, outputs, inPanel, outPanel, title},
 		If[ArgumentsQ["SchmidtCassegrainCAfter", {$SchmidtCassegrainCAfterInputVariables, {f1, ft, em, diam, theta}}],
			AppendTo[$ExamplesStack, 
					 <|"PackageID" -> 14, "PackageName" -> "SchmidtCassegrainCAfter", "Arguments" -> <|"f1" -> f1, "ft" -> ft, "em" -> em, "diam" -> diam, "\[Theta]" -> theta|>|>];
			$ExamplesStack = DeleteExampleDuplicates[$ExamplesStack];
			
			\[Alpha] = em/f1;
			M = ft/f1;
			\[Beta] = (M - \[Alpha])/(M + 1);
			\[Gamma] = -((2 M (1 + \[Alpha]))/(1 - M^2));
			
			(* Elimination of spherical aberration, coma, astigmatism, and chromatism*)
			rad = {-2 f1, -\[Gamma] f1, Infinity, Rc}; 
			thick = {-\[Beta] f1, \[Beta] f1, 5};
			ind = {{1, -1, 1, 1.518722, 1}, {1, -1, 1, 1.522376, 1}, {1, -1, 1, 1.514322, 1}};
			costasf = {K1, K2, 0, {a4, as}};
			TotalAberrations[rad, thick, ind, costasf, diam/2, 0, 0, -Infinity, x, theta, {\[Lambda]1, \[Lambda]2, \[Lambda]3}];
			s3 = Flatten@Solve[{GOSphericalCoefficient == 0, GOcoma[[1]] == 0, GOAstigmatismCoefficient == 0, GOdistancegauss[[2, GOn]] - GOdistancegauss[[3, GOn]] == 0}, {a4, K1, K2, Rc}];

			outputs = {	
						{"Primary radius", -N[2 f1]},
						{"Secondary radius", -N[\[Gamma] f1]},
						{"Distance between primary and secondary", -N[\[Beta] f1]},
						{"Distance between the secondary and the corrector", N[\[Beta] f1]},
						{"Conic constant of primary", ReplaceAll[K1, s3]},
						{"Conic constant of secondary", ReplaceAll[K2, s3]},
						{"Aspherical constant of corrector", ReplaceAll[a4, s3]},
						{"The radius of the second surface of corrector", ReplaceAll[Rc, s3]},
						{"The total focal length", GOfocalLength[[1]] /. s3},
						{"Height of image",  GOimageHeight[[1, GOn]] /. s3},
						{"Spherical aberration", GOaberration[[1]] /. s3},
						{"Coma", GOcoma[[1]] /. s3},
						{"Astigmatism", GOastigmatism[[1]] /. s3},
						{"Curvature", GOcurvature[[1]] /. s3}};
			
			(* defines the two panels, input parameters and output values *)
			inPanel = Grid[{{"Focal length of the primary mirror" , "f1", f1},
							{"Refractive index of corrector", "ft", ft},
							{"Back distance", "em", em},
							{"Diameter of the corrector", "diam", diam},
							{"Field angle  in degree", "\[Theta]", theta}},
							Alignment -> {{Left, Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"InputParameterBottom"}];
  
			outPanel = Grid[outputs, 
							Alignment -> {{Left, Right}, Center}, 
							Spacings -> {2, 1}, 
							Dividers -> Center, 
							FrameStyle -> LightGray,
							BaseStyle->{"OutputValueBottom"}];

			(* generates the type of output required *)
			Switch[OptionValue[OutputType],
					"Report",
					GenerateDocument[TemplateApply[$ReportTemplate, 
										Join[<|	"title" -> $GeometricOpticsPackagesList[SelectFirst[#PackageName == "SchmidtCassegrainCAfter" &], "Description"],
												"date" -> DateString[], 
												"function" -> "SchmidtCassegrainCAfter", 
												"outputlevel" -> OptionValue[OutputLevel],
												"inPanel" -> inPanel, 
												"outPanel" -> outPanel |>]]];,

					"Print",
					CellPrint[TextCell[TemplateApply[$PrintTemplate, 
											Join[<|	"title" -> title, 
													"date" -> DateString[], 
													"function" -> "SchmidtCassegrainCAfter", 
													"outputlevel" -> OptionValue[OutputLevel],
													"inPanel" -> inPanel, 
													"outPanel" -> outPanel |>]], "Text"]];,
					"Basic",
					CellPrint[TextCell[
					TemplateApply[$BasicTemplate, 
									Join[<| "outputlevel" -> OptionValue[OutputLevel],
											"inputs" -> {{"f1", f1},
														 {"ft", ft},
														 {"em", em},
														 {"diam", diam},
														 {"\[Theta]", theta}},
											"outputs" -> outputs |>]], "Output"]],
					_,
					CellPrint[TextCell[DeleteCases[outputs[[All, 2]], Alternatives["", Style[___]]], "Output"]]],

							
			(* Arguments are not correct *)
			MessageDialog["SchmidtCassegrainCAfter not executed, the number or the type of the arguments may be incorrect.", WindowTitle->"Warning: example not generated"];]];

SchmidtCassegrainCAfterInputPanel[]:=
DynamicModule[{outputtype = "Report", package = "SchmidtCassegrainCAfter"},
	examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
						Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
    examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], Alignment -> Center, Spacings -> {1, 1}, Dividers -> Center], FrameStyle -> LightGray] &, examplesAll];
	example = "Browse...";
	Panel[Column[{	DynamicWrapper[Style[NameFromPackageName[package] <> " Input Panel", "Subsection"], 
									If[NumericQ[example], {f1, ft, em, diam, theta} = ReplaceAll[$SchmidtCassegrainCAfterInputVariables, examplesAll[[example]]]]],
					Style["Insert values for each argument, then use Evaluate to run SchmidtCassegrainCAfter function", "Text"],
					Grid[{{"Focal length of the primary mirror" , "f1", Tooltip[InputField[Dynamic[f1], Alignment -> Center], "Insert the focal length of the primary mirror"]},
						  {"Total focal length", "ft", Tooltip[InputField[Dynamic[ft], Alignment -> Center], "Input the total focal length"]},
						  {"Back distance", "em", Tooltip[InputField[Dynamic[em], Alignment -> Center], "Insert the back distance"]},
						  {"Diameter of the primary mirror", "diam", Tooltip[InputField[Dynamic[diam], Alignment -> Center], "Input the value of diam"]},
						  {"Field angle in degrees", "\[Theta]", Tooltip[InputField[Dynamic[theta], Alignment -> Center], "Insert the value of \[Theta]"]}},
						Spacings -> {1, 0},
						Alignment -> {{Left, Left, Right}, Center}, 
						Dividers -> Center, 
						FrameStyle -> LightGray],
					OpenerView[{"Load an example from the archives (current session and saved DB)",
								Row[{Dynamic@PopupMenu[Dynamic[example], Thread[Rule[Range[Length[examples]], examples]], If[examples === {}, "No example saved", "Browse..."], FrameMargins -> 3, Alignment -> Center],
									 Spacer[5],
									 Button["Update list", (examplesAll = Join[	Cases[Values[$ExamplesStack], {_, package, arguments_} :> arguments], 
																				Cases[Normal@Values[$ExamplesArchive], {_, _, package, arguments_, _} :> arguments]];
															examples = Map[Framed[Grid[Transpose[KeyValueMap[List, Association[#]]], 
																						Alignment -> Center, 
																						Spacings -> {1, 1}, 
																						Dividers -> Center], 
																						FrameStyle -> LightGray]&, examplesAll];
															example = "Browse..."), Method -> "Queued"]}]}, 
								Alignment -> Center, 
								Spacings -> 1.5],
					Row[{"Define the type of output to generate", 
						 Spacer[5],
						 RadioButtonBar[Dynamic[outputtype], {"Report" -> Tooltip["Report", "Generates a new notebook reporting a summary of the calculation"], 
															  "Print" -> Tooltip["Print", "Print the table of the calculation done inside the current notebook"], 
															  "Basic" -> Tooltip["Basic", "Generate a list of computed output with label"], 
															  "Values" -> Tooltip["Values", "Return only the list of output values"]}]}],
					Row[{Button["Evaluate", ToExpression[package][Apply[Sequence, {f1, ft, em, diam, theta}], OutputType -> outputtype], Method -> "Queued"],
						 Button["Clear all", Map[Clear, Unevaluated[{f1, ft, em, diam, theta}]]]}]}, 
				Spacings -> 2, 
				Alignment -> Center],
	BaseStyle -> {InputFieldBoxOptions -> {FieldSize -> {15, 1}}}]];
			
  
  End[]
  EndPackage[]